<?php
/**
 * @package     joomLab.Plugin
 * @subpackage  joomLab.Avatar
 *
 * @copyright   (C) 2025 Alexandr Novikov. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */
defined('_JEXEC') or die;

use Joomla\CMS\Application\AdministratorApplication;
use Joomla\CMS\Factory;
use Joomla\CMS\Installer\Installer;
use Joomla\CMS\Installer\InstallerAdapter;
use Joomla\CMS\Installer\InstallerScriptInterface;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Version;
use Joomla\Database\DatabaseDriver;
use Joomla\DI\Container;
use Joomla\DI\ServiceProviderInterface;
use Joomla\CMS\Filesystem\File;
use Joomla\CMS\Filesystem\Folder;
use Joomla\CMS\Filesystem\Path;
use Joomla\CMS\Access\Access;
use Joomla\CMS\Table\Table;
use Joomla\CMS\Access\Rules;

return new class () implements ServiceProviderInterface {
	public function register(Container $container)
	{
		$container->set(InstallerScriptInterface::class, new class ($container->get(AdministratorApplication::class)) implements InstallerScriptInterface {
			protected AdministratorApplication $app;
			protected DatabaseDriver $db;
			protected string $minimumJoomla = '5.0';
			protected string $minimumPhp = '8.0';
			public function __construct(AdministratorApplication $app)
			{
				$this->app = $app;
				$this->db  = Factory::getContainer()->get('DatabaseDriver');
			}
			public function install(InstallerAdapter $adapter): bool
			{
				$this->enablePlugin($adapter);
				return true;
			}
			public function update(InstallerAdapter $adapter): bool
			{
				(new Version())->refreshMediaVersion();
				return true;
			}
			public function uninstall(InstallerAdapter $adapter): bool
			{
				return true;
			}
			public function preflight(string $type, InstallerAdapter $adapter): bool
			{
				if (!$this->checkCompatible())
				{
					return false;
				}
				return true;
			}
            public function postflight(string $type, InstallerAdapter $adapter): bool
            {
	            $installer = $adapter->getParent();
	            if ($type !== 'uninstall')
	            {
		            $this->parseLayouts($installer->getManifest()->layouts, $installer);
					$dir = JPATH_ROOT.'/images';
					$dir_media = JPATH_ROOT. '/media/plg_content_joomlab_gallery/images';
		            
		            if (Folder::exists($dir_media)) {
			            // Копируем всё содержимое из $dir_media в $dir
			            Folder::copy($dir_media, $dir, '', true);

			            // Удаляем содержимое исходной папки (опционально)
			             Folder::delete($dir_media);
		            }
	            }
	            else
	            {
		            $this->removeLayouts($installer->getManifest()->layouts);
	            }


                $wt_plugin_info = simplexml_load_file(JPATH_PLUGINS . '/editors-xtd/button_gallery/button_gallery.xml');
                $html = '<div class="wt-b24-plugin-info row">
				<div class="col-2 d-flex justify-content-center align-items-center">
				<a class="preloader_link" href="https://joomlab.ru" title="Расширения для CMS Joomla">
					<img width="150" src="https://joomlab.ru/images/logo_joomlab.png" alt="joomlab">
				</a>
				</div>
				<div class="col-10">
					<div class="p-1">
						<span class="badge bg-success">v.' . $wt_plugin_info->version . '</span>
					</div>
					<div class="card-body">
						Плагин кнопки добавления галерей Joomlab Gallery для CMS Joomla
					</div>
				</div>
			</div>';
                $this->app->enqueueMessage($html, 'info');
                return true;
            }

			public function parseLayouts(SimpleXMLElement $element = null, Installer $installer = null): bool
			{
				if (!$element || !count($element->children()))
				{
					return false;
				}
				$folder      = ((string) $element->attributes()->destination) ? '/' . $element->attributes()->destination : null;
				$destination = Path::clean(JPATH_ROOT . '/layouts' . $folder);
				$folder = (string) $element->attributes()->folder;
				$source = ($folder && file_exists($installer->getPath('source') . '/' . $folder))
					? $installer->getPath('source') . '/' . $folder : $installer->getPath('source');
				$copyFiles = [];
				foreach ($element->children() as $file)
				{
					$path['src']  = Path::clean($source . '/' . $file);
					$path['dest'] = Path::clean($destination . '/' . $file);
					$path['type'] = $file->getName() === 'folder' ? 'folder' : 'file';
					if (basename($path['dest']) !== $path['dest'])
					{
						$newdir = dirname($path['dest']);
						if (!Folder::create($newdir))
						{
							Log::add(Text::sprintf('JLIB_INSTALLER_ERROR_CREATE_DIRECTORY', $newdir), Log::WARNING, 'jerror');

							return false;
						}
					}

					$copyFiles[] = $path;
				}

				return $installer->copyFiles($copyFiles, true);
			}

			protected function removeLayouts(SimpleXMLElement $element = null): bool
			{
				if (!$element || !count($element->children()))
				{
					return false;
				}
				$files = $element->children();

				// Get source
				$folder = ((string) $element->attributes()->destination) ? '/' . $element->attributes()->destination : null;
				$source = Path::clean(JPATH_ROOT . '/layouts' . $folder);
				foreach ($files as $file)
				{
					$path = Path::clean($source . '/' . $file);
					if (is_dir($path))
					{
						$val = Folder::delete($path);
					}
					else
					{
						$val = File::delete($path);
					}

					if ($val === false)
					{
						Log::add('Failed to delete ' . $path, Log::WARNING, 'jerror');
						return false;
					}
				}

				if (!empty($folder))
				{
					Folder::delete($source);
				}

				return true;
			}
			protected function checkCompatible(): bool
			{
				$app = Factory::getApplication();
				if (!(new Version())->isCompatible($this->minimumJoomla))
				{
					$app->enqueueMessage(Text::sprintf('PLG_EDITORS_JS_WRONG_JOOMLA', $this->minimumJoomla),
						'error');
					return false;
				}
				if (!(version_compare(PHP_VERSION, $this->minimumPhp) >= 0))
				{
					$app->enqueueMessage(Text::sprintf('Минимальная версия PHP', $this->minimumPhp),
						'error');

					return false;
				}

				return true;
			}
			protected function enablePlugin(InstallerAdapter $adapter)
			{
				$plugin          = new \stdClass();
				$plugin->type    = 'plugin';
				$plugin->element = $adapter->getElement();
				$plugin->folder  = (string) $adapter->getParent()->manifest->attributes()['group'];
				$plugin->enabled = 1;
				$this->db->updateObject('#__extensions', $plugin, ['type', 'element', 'folder']);
			}
		});
	}
};