document.addEventListener('DOMContentLoaded', function() {
    let initializedSwitchers = new Set();
    function initSwitchers() {
        const modal = document.querySelector('.joomlab_modal');
        if (!modal) return;
        const switchers = modal.querySelectorAll('.switcher');
        switchers.forEach(switcher => {
            const switcherId = Array.from(switcher.querySelectorAll('input[type="radio"]'))
                .map(input => input.id || input.name)
                .join('-');
            if (!initializedSwitchers.has(switcherId)) {
                const row = switcher.closest('.row');
                if (!row) return;
                const radio0 = switcher.querySelector('input[type="radio"][value="0"]');
                const controls = row.querySelectorAll('.col-md-6:not(:has(.switcher)) select, .col-md-6:not(:has(.switcher)) input:not([type="radio"]):not([type="checkbox"])');
                if (radio0 && controls.length > 0) {
                    const updateState = () => {
                        controls.forEach(control => {
                            control.disabled = radio0.checked;
                        });
                    };
                    updateState();
                    const radios = switcher.querySelectorAll('input[type="radio"]');
                    radios.forEach(radio => {
                        radio.addEventListener('change', updateState);
                    });
                    initializedSwitchers.add(switcherId);
                }
            }
        });
    }
    initSwitchers();
    setInterval(initSwitchers, 1000);
    const observer = new MutationObserver(function() {
        initSwitchers();
    });
    observer.observe(document.body, {
        childList: true,
        subtree: true
    });
});

function initSwipers() {
    // Глобальный счетчик для уникальных индексов файлов
    if (typeof window.swiperFileCounter === 'undefined') {
        window.swiperFileCounter = 1000;
    }

    // Функция для обновления видимости thumbnails контейнеров
    function updateThumbnailsVisibility() {
        document.querySelectorAll('input[type="radio"][name*="[check_desc]"]:checked').forEach(function(radio) {
            const radioId = radio.id;
            const dynamicPart = radioId.replace(/^desc-/, '').replace(/[01]$/, '');
            const containerId = 'thumbnails_' + dynamicPart;

            const container = document.getElementById(containerId);
            if (container) {
                container.style.display = radio.value === '0' ? 'none' : 'block';
            }
        });
    }

    // Инициализация обработчиков для радиокнопок
    function initRadioButtons() {
        // Делегирование событий для радиокнопок
        document.addEventListener('change', function(e) {
            if (e.target && e.target.matches('input[type="radio"][name*="[check_desc]"]')) {
                updateThumbnailsVisibility();
            }
        });

        // Инициализация при загрузке
        updateThumbnailsVisibility();
    }

    const initSwiper = (container) => {
        if (container.classList.contains('swiper-initialized')) {
            if (container.swiper) {
                container.swiper.update();
            }
            return;
        }
        container.classList.add('swiper-initialized');
        new Swiper(container, {
            direction: 'horizontal',
            loop: false,
            pagination: {
                el: container.querySelector('.swiper-pagination'),
                clickable: true,
            },
            navigation: {
                nextEl: container.querySelector('.swiper-button-next'),
                prevEl: container.querySelector('.swiper-button-prev'),
            },
            "breakpoints": {
                "320": {
                    "slidesPerView": 1,
                    "spaceBetween": 15
                },
                "768": {
                    "slidesPerView": 3,
                    "spaceBetween": 15
                },
                "982": {
                    "slidesPerView": 5,
                    "spaceBetween": 15
                },
                "1200": {
                    "slidesPerView": 8,
                    "spaceBetween": 15
                }
            },
            slidesPerView: 'auto',
            watchOverflow: true,
            resistance: true,
            resistanceRatio: 0,
            observer: true,
            observeSlideChildren: true,
            observeParents: true
        });
    };

    const validateFiles = (newFiles, existingFileNames) => {
        const validFiles = [];
        const rejectedFiles = [];
        newFiles.forEach(file => {
            if (!file.type.startsWith('image/')) {
                rejectedFiles.push({ file, reason: 'not_image' });
                return;
            }
            if (existingFileNames.includes(file.name)) {
                rejectedFiles.push({ file, reason: 'duplicate' });
                return;
            }
            validFiles.push(file);
        });
        return { validFiles, rejectedFiles };
    };

    const showFileValidationMessage = (rejectedFiles) => {
        if (rejectedFiles.length === 0) return;
        const notImageCount = rejectedFiles.filter(f => f.reason === 'not_image').length;
        const duplicateCount = rejectedFiles.filter(f => f.reason === 'duplicate').length;
        let message = 'Не удалось добавить файлы:\n';
        if (notImageCount > 0) {
            message += `• ${notImageCount} не являются изображениями\n`;
        }
        if (duplicateCount > 0) {
            message += `• ${duplicateCount} уже добавлены ранее`;
        }
        alert(message);
    };

    const getExistingFileNamesFromSlides = (swiperWrapper) => {
        const removeButtons = swiperWrapper.querySelectorAll('.remove-image');
        return Array.from(removeButtons).map(button => button.dataset.filename);
    };

    const createThumbnailBlock = (file, slideIndex, controlsContainer, swiperId, uniqueIndex) => {
        let thumbnailContainer = controlsContainer.querySelector('.thumbnails-container');

        if (!thumbnailContainer) {
            thumbnailContainer = document.createElement('div');
            thumbnailContainer.className = 'thumbnails-container';
            thumbnailContainer.id = 'thumbnails_' + swiperId.replace('swiper-', '');
            controlsContainer.appendChild(thumbnailContainer);

            // После создания контейнера обновляем видимость
            setTimeout(updateThumbnailsVisibility, 50);
        }

        const baseName = swiperId.replace('swiper-', '');
        const cleanBaseName = baseName.replace(/^swiper_?/, '');
        const inputName = cleanBaseName.replace(/__/g, '][').replace(/_/g, '[') + '][description][' + uniqueIndex + ']';
        const inputId = cleanBaseName + `_description_${uniqueIndex}`;

        const thumbnailBlock = document.createElement('div');
        thumbnailBlock.className = 'thumbnail-block';
        thumbnailBlock.dataset.slideIndex = slideIndex;
        thumbnailBlock.dataset.fileIndex = uniqueIndex;
        thumbnailBlock.draggable = true;

        thumbnailBlock.innerHTML = `
            <button type="button" class="group-slide btn btn-sm btn-primary" aria-label="Переместить" data-slide-index="${slideIndex}">
                <span class="icon-arrows-alt icon-white" aria-hidden="true"></span>
            </button>
            <div class="thumbnail-preview">
                <img src="" alt="${file.name}">
            </div>
            <input type="text" 
                   class="form-control" 
                   name="${inputName}"
                   id="${inputId}"
                   placeholder="Описание" 
                   data-slide-index="${slideIndex}"
                   data-file-index="${uniqueIndex}"
            />
            <button type="button" class="remove-thumbnail" data-slide-index="${slideIndex}" data-file-index="${uniqueIndex}">×</button>
        `;

        const reader = new FileReader();
        reader.onload = function(e) {
            const img = thumbnailBlock.querySelector('img');
            img.src = e.target.result;
        };
        reader.readAsDataURL(file);
        thumbnailContainer.appendChild(thumbnailBlock);

        initDragAndDrop(thumbnailBlock, thumbnailContainer);
    };

    const initDragAndDrop = (thumbnailBlock, thumbnailContainer) => {
        thumbnailBlock.addEventListener('dragstart', function(e) {
            e.dataTransfer.effectAllowed = 'move';
            e.dataTransfer.setData('text/plain', this.dataset.fileIndex);
            this.classList.add('dragging');
        });

        thumbnailBlock.addEventListener('dragend', function(e) {
            this.classList.remove('dragging');
            const allBlocks = thumbnailContainer.querySelectorAll('.thumbnail-block');
            allBlocks.forEach(block => {
                block.classList.remove('drag-over-top', 'drag-over-bottom');
            });
        });

        thumbnailBlock.addEventListener('dragover', function(e) {
            e.preventDefault();
            e.dataTransfer.dropEffect = 'move';

            if (this.classList.contains('dragging')) {
                return;
            }

            const rect = this.getBoundingClientRect();
            const midpoint = rect.top + rect.height / 2;

            this.classList.remove('drag-over-top', 'drag-over-bottom');

            if (e.clientY < midpoint) {
                this.classList.add('drag-over-top');
            } else {
                this.classList.add('drag-over-bottom');
            }
        });

        thumbnailBlock.addEventListener('dragleave', function(e) {
            this.classList.remove('drag-over-top', 'drag-over-bottom');
        });

        thumbnailBlock.addEventListener('drop', function(e) {
            e.preventDefault();
            this.classList.remove('drag-over-top', 'drag-over-bottom');

            const draggedFileIndex = e.dataTransfer.getData('text/plain');
            const targetFileIndex = this.dataset.fileIndex;

            if (draggedFileIndex !== targetFileIndex) {
                const draggedElement = thumbnailContainer.querySelector(`.thumbnail-block[data-file-index="${draggedFileIndex}"]`);
                if (draggedElement) {
                    const rect = this.getBoundingClientRect();
                    const midpoint = rect.top + rect.height / 2;
                    const insertBefore = e.clientY < midpoint;

                    if (insertBefore) {
                        thumbnailContainer.insertBefore(draggedElement, this);
                    } else {
                        thumbnailContainer.insertBefore(draggedElement, this.nextSibling);
                    }

                    draggedElement.classList.add('dropped');
                    setTimeout(() => {
                        draggedElement.classList.remove('dropped');
                    }, 200);

                    updateFileOrder(thumbnailContainer);
                }
            }
        });
    };

    const updateFileOrder = (thumbnailContainer) => {
        const thumbnailBlocks = Array.from(thumbnailContainer.querySelectorAll('.thumbnail-block'));
        const fileInputs = Array.from(thumbnailContainer.closest('.controls').querySelectorAll('input[type="file"][data-file-index]'));

        thumbnailBlocks.forEach((block, newIndex) => {
            const fileIndex = block.dataset.fileIndex;
            block.dataset.slideIndex = newIndex;

            fileInputs.forEach(input => {
                if (input.dataset.fileIndex === fileIndex) {
                    const nameParts = input.name.split('[');
                    const baseName = nameParts.slice(0, -2).join('[');
                    input.name = `${baseName}[files][${newIndex}]`;
                }
            });

            const removeBtn = block.querySelector('.remove-thumbnail');
            const descriptionInput = block.querySelector('input.form-control');

            if (removeBtn) {
                removeBtn.dataset.slideIndex = newIndex;
            }

            if (descriptionInput) {
                descriptionInput.dataset.slideIndex = newIndex;
                const nameParts = descriptionInput.name.split('[');
                const baseNameDesc = nameParts.slice(0, -2).join('[');
                descriptionInput.name = `${baseNameDesc}[description][${fileIndex}]`;
                const idParts = descriptionInput.id.split('_');
                const baseId = idParts.slice(0, -1).join('_');
                descriptionInput.id = `${baseId}_${fileIndex}`;
            }
        });

        updateSwiperSlidesOrder(thumbnailContainer);
    };

    const updateSwiperSlidesOrder = (thumbnailContainer) => {
        const controlsContainer = thumbnailContainer.closest('.controls');
        const swiperContainer = controlsContainer.querySelector('.preview-swiper');
        if (!swiperContainer) return;

        const swiperWrapper = swiperContainer.querySelector('.swiper-wrapper');
        if (!swiperWrapper) return;

        const thumbnailBlocks = Array.from(thumbnailContainer.querySelectorAll('.thumbnail-block'));
        const fileIndicesOrder = thumbnailBlocks.map(block => parseInt(block.dataset.fileIndex));

        const slides = Array.from(swiperWrapper.querySelectorAll('.swiper-slide'));
        const slideMap = new Map();
        slides.forEach(slide => {
            const removeBtn = slide.querySelector('.remove-image');
            if (removeBtn && removeBtn.dataset.fileIndex !== undefined) {
                slideMap.set(parseInt(removeBtn.dataset.fileIndex), slide);
            }
        });

        swiperWrapper.innerHTML = '';

        fileIndicesOrder.forEach(fileIndex => {
            const slide = slideMap.get(fileIndex);
            if (slide) {
                swiperWrapper.appendChild(slide);
            }
        });

        if (swiperContainer.swiper) {
            setTimeout(() => {
                swiperContainer.swiper.update();
            }, 50);
        }
    };

    const removeThumbnailBlock = (fileIndex, controlsContainer) => {
        const thumbnailContainer = controlsContainer.querySelector('.thumbnails-container');
        if (!thumbnailContainer) return;
        const thumbnailBlock = thumbnailContainer.querySelector(`.thumbnail-block[data-file-index="${fileIndex}"]`);
        if (thumbnailBlock) {
            thumbnailBlock.remove();
        }
    };

    const createHiddenFileInput = (file, uniqueIndex, inputName) => {
        const hiddenInput = document.createElement('input');
        hiddenInput.type = 'file';
        const cleanInputName = inputName.replace(/^swiper_?/, '');
        hiddenInput.name = `${cleanInputName}[files][${uniqueIndex}]`;
        hiddenInput.style.display = 'none';
        hiddenInput.multiple = false;
        hiddenInput.dataset.fileIndex = uniqueIndex;
        hiddenInput.dataset.fileName = file.name;

        const dataTransfer = new DataTransfer();
        dataTransfer.items.add(file);
        hiddenInput.files = dataTransfer.files;

        return hiddenInput;
    };

    const handleFileSelect = async (event, input, swiperContainer, controlsContainer) => {
        const files = event.target.files;
        if (!files || !files.length) return;
        const swiperWrapper = swiperContainer.querySelector('.swiper-wrapper');
        if (!swiperWrapper) return;
        const existingFileNames = getExistingFileNamesFromSlides(swiperWrapper);
        const { validFiles, rejectedFiles } = validateFiles(Array.from(files), existingFileNames);
        showFileValidationMessage(rejectedFiles);
        if (validFiles.length === 0) {
            event.target.value = '';
            return;
        }

        const swiperId = swiperContainer.id;
        let baseInputName = input.name || swiperId.replace('swiper-', '').replace(/__/g, '][').replace(/_/g, '[');
        baseInputName = baseInputName.replace(/^swiper_?/, '');

        validFiles.forEach((file) => {
            const uniqueIndex = window.swiperFileCounter++;
            const hiddenInput = createHiddenFileInput(file, uniqueIndex, baseInputName);
            input.parentNode.appendChild(hiddenInput);
        });

        const loadPromises = validFiles.map((file, index) => {
            return new Promise((resolve) => {
                if (!file.type.startsWith('image/')) {
                    resolve(null);
                    return;
                }
                const reader = new FileReader();
                reader.onload = function(e) {
                    const slide = document.createElement('div');
                    slide.className = 'swiper-slide';
                    const uniqueIndex = window.swiperFileCounter - validFiles.length + index;
                    slide.innerHTML = `
                        <div class="image-preview">
                            <img src="${e.target.result}" alt="${file.name}">
                            <button type="button" class="remove-image" data-filename="${file.name}" data-file-index="${uniqueIndex}">×</button>
                        </div>
                    `;
                    swiperWrapper.appendChild(slide);
                    const totalSlides = swiperWrapper.querySelectorAll('.swiper-slide').length - 1;
                    createThumbnailBlock(file, totalSlides, controlsContainer, swiperId, uniqueIndex);
                    resolve(slide);
                };
                reader.onerror = () => resolve(null);
                reader.readAsDataURL(file);
            });
        });

        await Promise.all(loadPromises);
        if (!swiperContainer.classList.contains('swiper-initialized')) {
            initSwiper(swiperContainer);
        } else if (swiperContainer.swiper) {
            setTimeout(() => {
                if (swiperContainer.swiper) {
                    swiperContainer.swiper.update();
                    swiperContainer.swiper.slideTo(swiperContainer.swiper.slides.length - 1);
                }
            }, 50);
        }
        updateNavigationVisibility(swiperContainer);
        event.target.value = '';
    };

    const updateNavigationVisibility = (swiperContainer) => {
        const swiperWrapper = swiperContainer.querySelector('.swiper-wrapper');
        const hasSlides = swiperWrapper && swiperWrapper.querySelector('.swiper-slide');
        const pagination = swiperContainer.querySelector('.swiper-pagination');
        const nextBtn = swiperContainer.querySelector('.swiper-button-next');
        const prevBtn = swiperContainer.querySelector('.swiper-button-prev');
        if (pagination) {
            pagination.style.display = hasSlides ? 'block' : 'none';
        }
        if (nextBtn) {
            nextBtn.style.display = hasSlides ? 'flex' : 'none';
        }
        if (prevBtn) {
            prevBtn.style.display = hasSlides ? 'flex' : 'none';
        }
    };

    const removeImage = (slide, filename, fileIndex, input, swiperContainer, controlsContainer) => {
        if (!slide || !swiperContainer) return;

        const hiddenInput = input.parentNode.querySelector(`input[type="file"][data-file-index="${fileIndex}"]`);
        if (hiddenInput) {
            hiddenInput.remove();
        }

        const swiperWrapper = swiperContainer.querySelector('.swiper-wrapper');
        slide.remove();
        removeThumbnailBlock(fileIndex, controlsContainer);

        if (swiperContainer.swiper) {
            setTimeout(() => {
                if (swiperContainer.swiper) {
                    swiperContainer.swiper.update();
                    updateNavigationVisibility(swiperContainer);
                }
            }, 50);
        } else {
            updateNavigationVisibility(swiperContainer);
        }
    };

    const removeThumbnailAndSlide = (fileIndex, input, swiperContainer, controlsContainer) => {
        const swiperWrapper = swiperContainer.querySelector('.swiper-wrapper');
        const slides = swiperWrapper.querySelectorAll('.swiper-slide');

        let targetSlide = null;
        slides.forEach(slide => {
            const removeBtn = slide.querySelector('.remove-image');
            if (removeBtn && parseInt(removeBtn.dataset.fileIndex) === parseInt(fileIndex)) {
                targetSlide = slide;
            }
        });

        if (targetSlide) {
            const removeBtn = targetSlide.querySelector('.remove-image');
            const filename = removeBtn?.dataset.filename;

            if (filename && fileIndex !== undefined) {
                removeImage(targetSlide, filename, parseInt(fileIndex), input, swiperContainer, controlsContainer);
            }
        }
    };

    const initButtonHandler = (addButton) => {
        if (addButton._initialized) {
            return;
        }
        const controlsContainer = addButton.closest('.controls');
        if (!controlsContainer) return;
        const input = controlsContainer.querySelector('input[type="file"]');
        if (!input) return;
        const swiperContainer = controlsContainer.querySelector('.preview-swiper');
        if (!swiperContainer) return;
        const swiperWrapper = swiperContainer.querySelector('.swiper-wrapper');
        if (!swiperWrapper) return;

        const clickHandler = function(e) {
            e.preventDefault();
            e.stopPropagation();
            input.click();
        };

        const changeHandler = (event) => {
            handleFileSelect(event, input, swiperContainer, controlsContainer);
        };

        const removeHandler = (event) => {
            if (event.target.classList.contains('remove-image')) {
                const slide = event.target.closest('.swiper-slide');
                const filename = event.target.dataset.filename;
                const fileIndex = event.target.dataset.fileIndex;
                if (slide && filename && fileIndex !== undefined) {
                    removeImage(slide, filename, parseInt(fileIndex), input, swiperContainer, controlsContainer);
                }
                event.preventDefault();
                event.stopPropagation();
            }
        };

        const thumbnailRemoveHandler = (event) => {
            if (event.target.classList.contains('remove-thumbnail')) {
                const fileIndex = event.target.dataset.fileIndex;
                removeThumbnailAndSlide(fileIndex, input, swiperContainer, controlsContainer);
                event.preventDefault();
                event.stopPropagation();
            }
        };

        addButton.addEventListener('click', clickHandler);
        input.addEventListener('change', changeHandler);
        swiperContainer.addEventListener('click', removeHandler);

        let thumbnailContainer = controlsContainer.querySelector('.thumbnails-container');
        if (!thumbnailContainer) {
            thumbnailContainer = document.createElement('div');
            thumbnailContainer.className = 'thumbnails-container';
            thumbnailContainer.id = 'thumbnails_' + swiperContainer.id.replace('swiper-', '');
            controlsContainer.appendChild(thumbnailContainer);
        }
        thumbnailContainer.addEventListener('click', thumbnailRemoveHandler);
        updateNavigationVisibility(swiperContainer);
        addButton._initialized = true;

        if (swiperWrapper.querySelector('.swiper-slide') && !swiperContainer.classList.contains('swiper-initialized')) {
            initSwiper(swiperContainer);
        }
    };

    const initExistingThumbnails = () => {
        const thumbnailBlocks = document.querySelectorAll('.thumbnail-block:not([data-drag-initialized])');
        thumbnailBlocks.forEach(block => {
            const thumbnailContainer = block.closest('.thumbnails-container');
            if (thumbnailContainer) {
                initDragAndDrop(block, thumbnailContainer);
                block.setAttribute('data-drag-initialized', 'true');
            }
        });
    };

    const initAllButtons = () => {
        const addButtons = document.querySelectorAll('.add-image-btn');
        addButtons.forEach((button) => {
            initButtonHandler(button);
        });
        initExistingThumbnails();
    };

    const initExistingSwipers = () => {
        const existingSwipers = document.querySelectorAll('.preview-swiper');
        existingSwipers.forEach(container => {
            if (!container.classList.contains('swiper-initialized')) {
                const wrapper = container.querySelector('.swiper-wrapper');
                if (wrapper && wrapper.querySelector('.swiper-slide')) {
                    initSwiper(container);
                }
            }
            updateNavigationVisibility(container);
        });
        initExistingThumbnails();
    };

    // Инициализируем обработчики радиокнопок
    initRadioButtons();

    if (document.body) {
        if (window.swiperObserver) {
            window.swiperObserver.disconnect();
        }
        const observer = new MutationObserver((mutations) => {
            let shouldReinit = false;
            mutations.forEach((mutation) => {
                if (mutation.addedNodes.length) {
                    mutation.addedNodes.forEach((node) => {
                        if (node.nodeType === 1) {
                            const newSwipers = node.querySelectorAll('.preview-swiper');
                            if (newSwipers.length > 0) {
                                newSwipers.forEach(container => {
                                    if (!container.classList.contains('swiper-initialized')) {
                                        const wrapper = container.querySelector('.swiper-wrapper');
                                        if (wrapper && wrapper.querySelector('.swiper-slide')) {
                                            initSwiper(container);
                                        }
                                    }
                                    updateNavigationVisibility(container);
                                });
                                shouldReinit = true;
                            }
                            const newButtons = node.querySelectorAll('.add-image-btn');
                            if (newButtons.length > 0) {
                                shouldReinit = true;
                            }
                            if (node.matches('.controls')) {
                                shouldReinit = true;
                            }
                            const newThumbnails = node.querySelectorAll('.thumbnail-block');
                            if (newThumbnails.length > 0) {
                                shouldReinit = true;
                            }
                            // Проверяем новые радиокнопки
                            const newRadios = node.querySelectorAll('input[type="radio"][name*="[check_desc]"]');
                            if (newRadios.length > 0) {
                                shouldReinit = true;
                            }
                        }
                    });
                }
            });
            if (shouldReinit) {
                setTimeout(() => {
                    initAllButtons();
                    initExistingSwipers();
                    updateThumbnailsVisibility(); // Обновляем видимость при изменениях
                }, 100);
            }
        });
        observer.observe(document.body, {
            childList: true,
            subtree: true
        });

        window.swiperObserver = observer;
    }

    setTimeout(() => {
        initAllButtons();
        initExistingSwipers();
        updateThumbnailsVisibility(); // Инициализация при загрузке
    }, 100);
}

if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initSwipers);
} else {
    initSwipers();
}

window.reinitSwipers = function() {
    if (typeof initSwipers === 'function') {
        initSwipers();
    }
};
